package com.wudsn.productions.atari800.badapplehd;

import java.io.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

final class Frame {
    public final Screen screen;
    private int usedTilesCount;

    public int[] tileIDs;
    public List<Tile> tilesList;
    public Map<Tile, Long> tileCountMap;

    public byte[] screenMemory;
    public int[] tileSetTable;
    public List<TileSet> tileSets;

    public byte[] soundMemory;

    public Frame(FrameSequence frameSequence) {
	this.screen = frameSequence.screen;
	tilesList = new ArrayList<Tile>();
	tileCountMap = new TreeMap<Tile, Long>();
	soundMemory = new byte[frameSequence.sound.soundMemorySize];
    }

    public void readFrom(DataInputStream screenDataInputStream,
	    DataInputStream soundDataInputStream) throws IOException {
	if (screenDataInputStream == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'screenDataInputStream' must not be null.");
	}
	if (soundDataInputStream == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'soundDataInputStream' must not be null.");
	}
	tileIDs = new int[screen.tiles];
	for (int i = 0; i < tileIDs.length; i++) {
	    tileIDs[i] = screenDataInputStream.readInt();
	}
	usedTilesCount = screenDataInputStream.readInt();
	int usedTilesBytes = usedTilesCount * screen.bytesPerTile;
	byte[] tileMemory = new byte[usedTilesBytes];
	screenDataInputStream.read(tileMemory, 0, usedTilesBytes);
	for (int i = 0; i < usedTilesCount; i++) {
	    Tile tile = new Tile(tileMemory, i * screen.bytesPerTile,
		    screen.bytesPerTile);
	    addTile(tile);
	}
	soundDataInputStream.read(soundMemory);

    }

    public void addTile(Tile tile) {
	tilesList.add(tile);
	tileCountMap.put(tile, Long.valueOf(0));

	Long count = tileCountMap.get(tile);
	if (count == null) {
	    count = Long.valueOf(1);
	} else {
	    count = Long.valueOf(count.longValue() + 1);
	}
	tileCountMap.put(tile, count);
    }

    public int getUsedTilesCount() {
	return usedTilesCount;
    }

    public void convert() {

	// Convert tiles to characters.
	screenMemory = new byte[tileIDs.length];
	tileSetTable = new int[screen.rows];

	int maxTilesPerTileSet = 128;
	tileSets = new ArrayList<TileSet>();
	TileSet tileSet = new TileSet(screen.bytesPerTile);
	tileSets.add(tileSet);
	for (int row = 0; row < screen.rows; row++) {
	    int rowOffset = row * screen.columns;

	    // Analyze next row.
	    TileSet rowTileSet = new TileSet(tileSet);
	    for (int column = 0; column < screen.columns; column++) {
		int offset = rowOffset + column;
		int tileNumber = tileIDs[offset];
		if (tileNumber >= 0) {
		} else {
		    tileNumber = -(tileNumber + 1);
		}
		Tile tile = tilesList.get(tileNumber);
		rowTileSet.addTile(tile);
	    }

	    // Will the tile set exceed the available number of characters?
	    if (rowTileSet.getSize() > maxTilesPerTileSet) {
		tileSet = new TileSet(screen.bytesPerTile);
		tileSets.add(tileSet);
		row = row - 1; // Backtracking to same row with new empty
		// tile set
		continue;
	    }
	    tileSet.addTiles(rowTileSet);

	    // Row analysis completed, all tiles mapped.
	    tileSetTable[row] = tileSets.size() - 1;

	    for (int column = 0; column < screen.columns; column++) {
		int offset = rowOffset + column;
		int tileNumber = tileIDs[offset];

		// Note that the mask is inverted, because of the PM
		// underlay.
		int mask;
		if (tileNumber >= 0) {
		    mask = 0x80;
		} else {
		    tileNumber = -(tileNumber + 1);
		    mask = 0x00;
		}
		Tile tile = tilesList.get(tileNumber);
		int screenTileNumber = tileSet.getTileNumber(tile);

		screenMemory[offset] = (byte) (screenTileNumber ^ mask);

	    }
	}
    }

    /**
     * If there are more then one tile sets, then every tile set except the last
     * one must be padded to fill a complete charset.
     * 
     * @param index
     *            Index in tileSets, a non-negative integer.
     * @param charsetSize
     *            The size of a full charset in bytes
     * @return The tile as bytes, not <code>null</code>
     */
    public byte[] getTilesAsBytesWithPadding(int index, int charsetSize) {
	int size = tileSets.size();
	byte[] result = tileSets.get(index).getTilesAsBytes();
	if (size > 1 && index < size - 1 && result.length < charsetSize) {
	    byte[] charset = new byte[charsetSize];
	    System.arraycopy(result, 0, charset, 0, result.length);
	    result = charset;
	}
	return result;
    }

    public boolean contentEquals(Frame otherFrame) {
	if (otherFrame == null) {
	    return false;
	}
	if (!Arrays.equals(screenMemory, otherFrame.screenMemory)) {
	    return false;
	}
	if (!Arrays.equals(tileSetTable, otherFrame.tileSetTable)) {
	    return false;
	}
	if (tileSets.size() != otherFrame.tileSets.size()) {
	    return false;
	}
	for (int i = 0; i < tileSets.size(); i++) {
	    if (!Arrays.equals(tileSets.get(i).getTilesAsBytes(),
		    otherFrame.tileSets.get(i).getTilesAsBytes())) {
		return false;
	    }
	}
	return true;
    }
}