package com.wudsn.productions.atari800.badapplehd;

import java.io.DataOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Arrays;

import com.wudsn.gfx.avi.AVIImageWriter;
import com.wudsn.gfx.avi.AVIReader.CHUNK;

/**
 * 
 * @author Peter Dell
 * 
 */
class BadAppleImageWriter extends AVIImageWriter {
    private static final int PIXELS_PER_COLUMN = 8;
    private static final int LINES_PER_ROW = 8;

    private static final int MAX_COLUMS_PER_LINE = 48;

    int columnsPerLine;
    int rowsPerScreen;
    int tilesPerScreen;
    private int[] screenMemory;

    private byte[][] tileMemory;
    private int usedTileCount;

    private boolean useInverseTiles;
    private boolean[] inverseTilesUsed;
    private int inverseTilesCount;

    public BadAppleImageWriter() {

    }

    @Override
    public void initialize(int imageWidth, int imageHeight,
	    double imageWidthFactor, double imageHeightFactor) {

	super.initialize(imageWidth, imageHeight, imageWidthFactor,
		imageHeightFactor);

	int pixelWidth = scaledWidth;
	int pixelHeight = scaledHeight;

	columnsPerLine = (pixelWidth + PIXELS_PER_COLUMN - 1)
		/ PIXELS_PER_COLUMN;
	columnsPerLine = Math.max(columnsPerLine, MAX_COLUMS_PER_LINE);
	rowsPerScreen = (pixelHeight + LINES_PER_ROW - 1) / LINES_PER_ROW;

	tilesPerScreen = columnsPerLine * rowsPerScreen;

	screenMemory = new int[tilesPerScreen];
	tileMemory = new byte[tilesPerScreen][];

	useInverseTiles = true;
	for (int i = 0; i < tilesPerScreen; i++) {
	    tileMemory[i] = new byte[LINES_PER_ROW];
	}
	inverseTilesUsed = new boolean[tilesPerScreen];
    }

    @Override
    public void convert(CHUNK imageChunk) {
	int threshold = 16;

	if (imageChunk == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'imageChunk' must not be null.");
	}
	byte[] image = imageChunk.data;
	if (image.length > imageChunkSize) {
	    throw new IllegalArgumentException(
		    "ERROR: Image chunk has "
			    + image.length
			    + " bytes and is larger than the defined video chunk size of "
			    + imageChunkSize + " bytes.");
	}

	Arrays.fill(screenMemory, 0);
	usedTileCount = 1; // Tile 00 is reserved for space
	Arrays.fill(inverseTilesUsed, false);

	inverseTilesCount = 0;

	int[] rgb = new int[3];
	byte[] tileData = new byte[LINES_PER_ROW];
	byte[] tileDataInverse = new byte[LINES_PER_ROW];

	int screenMemoryIndex = 0;
	for (int row = 0; row < rowsPerScreen; row++) {
	    int yStart = row * LINES_PER_ROW;
	    for (int column = 0; column < columnsPerLine; column++) {
		int xStart = column * PIXELS_PER_COLUMN;
		for (int y = 0; y < LINES_PER_ROW; y++) {
		    int m = 0x80;
		    int b = 0;
		    for (int x = 0; x < PIXELS_PER_COLUMN; x++) {
			getRGB(imageChunk, xStart + x, yStart + y, rgb);
			if (rgb[0] >= threshold) {
			    b |= m;
			}
			m = m >>> 1;
		    }
		    tileData[y] = (byte) b;
		    tileDataInverse[y] = (byte) ~tileData[y];
		}
		int tileFound = 0;
		int tileNumber = 0;
		while (tileFound == 0 && tileNumber < usedTileCount) {
		    if (Arrays.equals(tileData, tileMemory[tileNumber])) {
			tileFound = 1;
		    } else if (useInverseTiles
			    && Arrays.equals(tileDataInverse,
				    tileMemory[tileNumber])) {
			tileFound = -1;

			// Count unique inverse matches separately
			if (!inverseTilesUsed[tileNumber]) {
			    inverseTilesUsed[tileNumber] = true;
			    inverseTilesCount++;
			}

		    } else {
			tileNumber++;
		    }
		}
		if (tileFound == 0) {
		    System.arraycopy(tileData, 0, tileMemory[tileNumber], 0,
			    tileData.length);
		    tileFound = 1;
		    usedTileCount++;
		}

		if (tileFound == 1) {
		    screenMemory[screenMemoryIndex] = tileNumber;
		} else {
		    screenMemory[screenMemoryIndex] = -(tileNumber + 1);
		}
		screenMemoryIndex++;
	    }

	}
    }

    public void saveBuffersHeader(DataOutputStream outputStream)
	    throws IOException {

	outputStream.writeInt(columnsPerLine);
	outputStream.writeInt(rowsPerScreen);
	outputStream.writeInt(tilesPerScreen);
    }

    /**
     * Save the buffers to the stream.
     * 
     * @param outputStream
     *            The output stream, not <code>null</code>.
     * @throws IOException
     *             If an IO error occurs.
     */
    public void saveBuffers(DataOutputStream outputStream) throws IOException {
	if (outputStream == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'outputStream' must not be null.");
	}

	for (int i = 0; i < screenMemory.length; i++) {
	    outputStream.writeInt(screenMemory[i]);
	}

	outputStream.writeInt(usedTileCount);
	for (int i = 0; i < usedTileCount; i++) {
	    outputStream.write(tileMemory[i]);

	}
    }

    public void saveStatisticsHeader(PrintStream printStream) {
	printStream.print("usedTileCount");
	printStream.print(";");
	printStream.print("inverseTilesCount");

    }

    public void saveStatistics(PrintStream printStream) {
	printStream.print(usedTileCount);
	printStream.print(";");

	printStream.print(inverseTilesCount);

    }

}