package com.wudsn.productions.atari800.badapplehd;

import java.io.BufferedOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;

import com.wudsn.gfx.avi.AVIFrameConverter;
import com.wudsn.gfx.avi.AVIReader.AVIHeader;
import com.wudsn.gfx.avi.AVIReader.CHUNK;
import com.wudsn.gfx.avi.Log;

public final class BadAppleHDFrameConverter extends AVIFrameConverter {

    // Media parameters
    public static final int MEDIA_AUDIO_CHUNK_SIZE = 1280;
    public static final int MEDIA_AUDIO_CHANNELS = 2;

    private DataOutputStream headerDataOutputStream;
    private DataOutputStream tileDataOutputStream;
    private DataOutputStream audioDataOutputStream;

    private PrintStream statisticsPrintStream;

    private BadAppleImageWriter imageWriter;
    private BadAppleHDAudioWriter audioWriter;

    private int outputFrameCount;

    private long maxMediaFrameCount;

    public BadAppleHDFrameConverter() {
	imageWriter = new BadAppleImageWriter();
	audioWriter = new BadAppleHDAudioWriter(MEDIA_AUDIO_CHUNK_SIZE,
		MEDIA_AUDIO_CHANNELS, 8192, 312);

    }

    @Override
    public int init(Log log, AVIHeader aviHeader) {
	super.init(log, aviHeader);

	imageWriter.initialize(aviHeader.width, aviHeader.height, 1.0, 1.0);

	maxMediaFrameCount = Long.MAX_VALUE;
	frameCountSkip = 0; // 1; // Integer.MAX_VALUE;

	videoFrameCount = 0;
	outputFrameCount = 0;

	return MEDIA_AUDIO_CHUNK_SIZE;

    }

    @Override
    public void open(String outputFilePath) throws IOException {
	if (outputFilePath == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'outputFilePath' must not be null.");
	}
	this.outputFilePath = outputFilePath;
	headerDataOutputStream = new DataOutputStream(createOutputStream(
		outputFilePath, ".hdr"));
	tileDataOutputStream = new DataOutputStream(createOutputStream(
		outputFilePath, ".til"));
	audioDataOutputStream = new DataOutputStream(createOutputStream(
		outputFilePath, ".aud"));
	statisticsPrintStream = new PrintStream(createOutputStream(
		outputFilePath, ".csv"));

    }

    private static BufferedOutputStream createOutputStream(String filePath,
	    String suffix) throws FileNotFoundException {
	return new BufferedOutputStream(new FileOutputStream(new File(filePath
		+ suffix)));
    }

    @Override
    public void close() {
	if (tileDataOutputStream != null) {
	    try {
		tileDataOutputStream.close();
	    } catch (IOException ex) {
		throw new RuntimeException("Cannot close file", ex);
	    }
	}
	if (headerDataOutputStream != null) {
	    try {
		headerDataOutputStream.close();
	    } catch (IOException ex) {
		throw new RuntimeException("Cannot close file", ex);
	    }
	}
	statisticsPrintStream.close();
    }

    @Override
    public void saveHeaders() throws IOException {

	imageWriter.saveBuffersHeader(headerDataOutputStream);

	statisticsPrintStream.print("mediaFrameCount");
	statisticsPrintStream.print(";");
	statisticsPrintStream.print("outputFrameCount");
	statisticsPrintStream.print(";");
	imageWriter.saveStatisticsHeader(statisticsPrintStream);
	
	log.printLong("columnsPerLine", imageWriter.columnsPerLine);
	log.printLong("rowsPerScreen", imageWriter.rowsPerScreen);
	log.printLong("tilesPerScreen", imageWriter.tilesPerScreen);

	statisticsPrintStream.println();
    }

    @Override
    public void consume(int imageChunksCount, CHUNK imageChunk,
	    int audioChunksCount, CHUNK audioChunk) throws IOException {

	videoFrameCount = videoFrameCount + 1;
	log.logDetails("Video Frame " + videoFrameCount + ":");

	if (videoFrameCount > maxMediaFrameCount) {
	    return;
	}

	if (frameCountSkip == 0 || (videoFrameCount - 1) % frameCountSkip == 0) {

	    outputFrameCount = outputFrameCount + 1;
	    imageWriter.convert(imageChunk);
	    imageWriter.saveBuffers(tileDataOutputStream);

	    if (audioChunk != null) {
		audioWriter.convert(audioChunk.data);
		audioWriter.saveBuffer(audioDataOutputStream);
	    }

	    statisticsPrintStream.print(videoFrameCount);
	    statisticsPrintStream.print(";");
	    statisticsPrintStream.print(outputFrameCount);
	    statisticsPrintStream.print(";");
	    imageWriter.saveStatistics(statisticsPrintStream);
	    statisticsPrintStream.println();

	}

    }

    @Override
    public void logSummary() {
	// log.printLong("Audio MinValue", audioWriter.getMinValue());
	// log.printLong("Audio MaxValue", audioWriter.getMaxValue());
    }

}