package com.wudsn.gfx.avi;

import java.io.IOException;
import java.io.InputStream;

public final class AVIReader {

    public static abstract class ATOM {
	public long position;
	public String fourCC;
	public int size;
	public byte[] data;

	protected ATOM(long position) {
	    this.position = position;
	    fourCC = "????";
	    size = 0;
	    data = null;
	}

	@Override
	public String toString() {
	    return "position=0x" + Long.toHexString(position) + ",fourCC="
		    + fourCC + ",size=" + size;
	}

    }

    public static class CHUNK extends ATOM {

	public CHUNK(long position) {
	    super(position);

	}

	@Override
	public String toString() {
	    return "position=0x" + Long.toHexString(position)
		    + " CHUNK: fourCC=" + fourCC + " size=" + "0x"
		    + Integer.toHexString(size);
	}

    }

    public static class LIST extends ATOM {
	String list;

	public LIST(long position) {
	    super(position);
	    list = "????";
	}

	@Override
	public String toString() {
	    return "position=0x" + Long.toHexString(position) + " LIST: list="
		    + list + " fourCC=" + fourCC + " size="
		    + Integer.toHexString(size);
	}

    }

    public static class AVIHeader {
	public int microSecPerFrame; // frame display rate (or 0)
	public int maxBytesPerSec; // max. transfer rate
	public int paddingGranularity; // pad to multiples of this
	// size;
	public int flags; // the ever-present flags
	public int totalFrames; // # frames in file
	public int initialFrames;
	public int streams;
	public int suggestedBufferSize;
	public int width;
	public int height;
	int[] reserved = new int[4];

	@Override
	public String toString() {
	    return "AVIHeader: microSecPerFrame=" + microSecPerFrame
		    + " maxBytesPerSec=" + maxBytesPerSec
		    + " paddingGranularity=" + paddingGranularity + " flags="
		    + Integer.toHexString(flags) + " totalFrames="
		    + totalFrames + " initialFrames=" + initialFrames
		    + " streams=" + streams + " suggestedBufferSize="
		    + suggestedBufferSize + " width=" + width + " height="
		    + height;
	}
    }

    private InputStream inputStream;
    private long position;
    private byte[] dwordBuffer;

    public AVIReader(InputStream inputStream) {
	if (inputStream == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'inputStream' must not be null.");
	}
	this.inputStream = inputStream;
	position = 0;
	dwordBuffer = new byte[4];
    }

    public ATOM readAtom() {

	String fourCC = readFourCC();
	if (fourCC == null) {
	    return null;
	}
	if (fourCC.equals("LIST")) {
	    LIST list = new LIST(position);
	    list.list = fourCC;
	    list.size = readDWORD();
	    list.fourCC = readFourCC();
	    return list;
	}
	CHUNK chunk = new CHUNK(position);
	chunk.fourCC = fourCC;
	chunk.size = readDWORD();
	chunk.size = (chunk.size + 1) & 0xfffffffe;
	return chunk;
    }

    public void skipAtomData(ATOM atom) {
	if (atom == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'atom' must not be null.");
	}

	int size = atom.size;
	if (atom instanceof LIST) {
	    size = size - 4;
	}
	try {
	    inputStream.skip(size);
	    position += size;
	} catch (IOException ex) {
	    throw new RuntimeException("Cannot skip " + atom.size
		    + " bytes of atom '" + atom.fourCC + "'.");
	}
    }

    public CHUNK readChunk() {

	String fourCC = readFourCC();
	if (fourCC == null) {
	    return null;
	}
	CHUNK result = new CHUNK(position);

	result.fourCC = fourCC;
	result.size = readDWORD();

	return result;
    }

    public CHUNK readChunk(String fourCC) {
	if (fourCC == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'fourCC' must not be null.");
	}
	CHUNK result = readChunk();

	if (!fourCC.equals(result.fourCC)) {
	    throw new RuntimeException("FourCC is '" + result.fourCC
		    + "' instead of '" + fourCC + "'.");
	}

	return result;
    }

    public void readChunkData(CHUNK chunk) {
	if (chunk == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'chunk' must not be null.");
	}
	chunk.data = new byte[chunk.size];
	try {
	    inputStream.read(chunk.data);
	    position += chunk.size;
	} catch (IOException ex) {
	    throw new RuntimeException("Cannot read " + chunk.size
		    + " bytes of chunk '" + chunk.fourCC + "'.");
	}

    }

    public LIST readList() {
	String foundCC = readFourCC();
	if (foundCC == null) {
	    return null;
	}

	LIST result = new LIST(position);

	result.list = foundCC;
	result.size = readDWORD();
	result.fourCC = readFourCC();

	return result;
    }

    public void readListData(LIST list) {
	if (list == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'list' must not be null.");
	}
	try {

	    list.data = new byte[list.size - 4];
	    inputStream.read(list.data);

	} catch (IOException ex) {
	    throw new RuntimeException("Cannot read " + (list.size - 4)
		    + " bytes of list '" + list.fourCC + "'.");
	}
    }

    public AVIHeader readHeader() {
	CHUNK chunk = readChunk("avih");
	AVIHeader result = new AVIHeader();
	result.microSecPerFrame = readDWORD();
	result.maxBytesPerSec = readDWORD();
	result.paddingGranularity = readDWORD();
	result.flags = readDWORD();
	result.totalFrames = readDWORD();
	result.initialFrames = readDWORD();
	result.streams = readDWORD();
	result.suggestedBufferSize = readDWORD();
	result.width = readDWORD();
	result.height = readDWORD();
	result.reserved[0] = readDWORD();
	result.reserved[1] = readDWORD();
	result.reserved[2] = readDWORD();
	result.reserved[3] = readDWORD();
	position += chunk.size;
	return result;
    }

    public String readFourCC() {
	try {
	    int bytes = inputStream.read(dwordBuffer);
	    if (bytes == -1) {
		return null;

	    }
	    if (bytes != 4) {
		throw new RuntimeException("Cannot read DWORD");
	    }
	    position += bytes;
	} catch (IOException ex) {
	    throw new RuntimeException(ex);
	}
	return String.valueOf((char) dwordBuffer[0])
		+ String.valueOf((char) dwordBuffer[1])
		+ String.valueOf((char) dwordBuffer[2])
		+ String.valueOf((char) dwordBuffer[3]);
    }

    private int readDWORD() {
	int result;
	int FF = 0xff;
	int bytes;
	try {
	    bytes = inputStream.read(dwordBuffer);
	    if (bytes == -1) {
		throw new RuntimeException("End of file");
	    }
	    if (bytes != 4) {
		throw new RuntimeException("Cannot read DWORD");
	    }
	} catch (IOException ex) {
	    throw new RuntimeException(ex);
	}

	result = (dwordBuffer[3] & FF);
	result = result << 8;
	result += (dwordBuffer[2] & FF);
	result = result << 8;
	result += (dwordBuffer[1] & FF);
	result = result << 8;
	result += (dwordBuffer[0] & FF);

	position += bytes;
	return result;
    }

}