package com.wudsn.gfx.avi;

import java.awt.Color;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import com.wudsn.gfx.avi.AVIReader.CHUNK;

public abstract class AVIImageWriter {

    protected int[][] palette;
    protected boolean usePaletteBrightness;
    protected Map<Integer, Integer> colorMap;
    protected int imageWidth;
    protected int imageHeight;
    protected int imageChunkSize;
    protected double imageWidthFactor;
    protected double imageHeightFactor;
    protected int scaledWidth;
    protected int scaledHeight;

    protected AVIImageWriter() {
	palette = new int[256][3];
	colorMap = new HashMap<Integer, Integer>();
    }

    public void readPalette(String fileName, float saturationFactor,
	    float brighnessFactor) {

	usePaletteBrightness = true;

	byte[] d = new byte[768];
	try {
	    FileInputStream f = new FileInputStream(new File(fileName));
	    f.read(d);
	    f.close();
	} catch (IOException ex) {
	    throw new RuntimeException(ex);
	}
	int i = 0, y, r, g, b, rgb;
	float[] hsb = new float[3];
	for (y = 0; y < 256; y++) {
	    r = d[i++] & 0xFF;
	    g = d[i++] & 0xFF;
	    b = d[i++] & 0xFF;
	    Color.RGBtoHSB(r, g, b, hsb);
	    hsb[2] = Math.min(1.0f, hsb[2] * brighnessFactor);
	    rgb = Color.HSBtoRGB(hsb[0], hsb[1] / saturationFactor, hsb[2]);
	    palette[y][0] = (rgb >> 16) & 0xFF;
	    palette[y][1] = (rgb >> 8) & 0xFF;
	    palette[y][2] = (rgb) & 0xFF;
	}

	colorMap.clear(); // Palette has changed
    }

    public void initialize(int imageWidth, int imageHeight,
	    double imageWidthFactor, double imageHeightFactor) {

	this.imageWidth = imageWidth;
	this.imageHeight = imageHeight;
	this.imageChunkSize = imageWidth * imageWidth * 3;

	this.imageWidthFactor = imageWidthFactor;
	this.imageHeightFactor = imageHeightFactor;

	scaledWidth = (int) (this.imageWidth * imageWidthFactor);
	scaledHeight = (int) (this.imageHeight * imageHeightFactor);

	colorMap.clear(); // Image has changed
    }

    protected final void getRGB(CHUNK imageChunk, int x, int y, int[] rgb) {

	if (imageChunk.fourCC.equals("00dc")) {
	    y = imageHeight - 1 - y;
	}
	if (x >= scaledWidth || y > scaledHeight) {
	    rgb[0] = 0;
	    rgb[1] = 0;
	    rgb[2] = 0;
	    return;
	}
	int scaledx = (int) (x / imageWidthFactor);
	int scaledy = (int) ((scaledHeight - 1 - y) / imageHeightFactor);

	int offset = (scaledx + scaledy * imageWidth) * 3;
	try {
	    byte[] image=imageChunk.data;
	    rgb[0] = image[offset + 2] & 0xFF; // R
	    rgb[1] = image[offset + 1] & 0xFF; // G
	    rgb[2] = image[offset + 0] & 0xFF; // B
	} catch (ArrayIndexOutOfBoundsException ex) {
	    // throw new IllegalArgumentException("Position x=" + x + " y="
	    // + y + " is out of range.");
	    rgb[0] = 0;
	    rgb[1] = 0;
	    rgb[2] = 0;
	}
	// System.err.println(Arrays.toString(rgb));
    }

    public abstract void convert(CHUNK imageChunk);

}