package com.wudsn.gfx.avi;

// Converts AVI to Tiled.bin.
// Run com.wudsn.gfx.avi.AVIConverter with arguments:
// - C:\jac\system\Atari800\Programming\Demos\BadAppleHD\video BA-384x216@24.9305.avi
// - BA-384x216@24.9305-Tiled.bin
// - com.wudsn.productions.atari800.badapplehd.avi.FrameConverter
//
// Use "Nearest Neighbour" to keep the colors exactly as they are when resizing using VirtualDUB

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.wudsn.gfx.avi.AVIReader.ATOM;
import com.wudsn.gfx.avi.AVIReader.AVIHeader;
import com.wudsn.gfx.avi.AVIReader.CHUNK;
import com.wudsn.gfx.avi.AVIReader.LIST;

public final class AVIConverter {

    private final Log log;
    private final String videoFilePath;
    private final String outputFilePathPrefix;
    private final AVIFrameConverter frameConverter;

    private InputStream videoInputStream;
    private int mediaAudioChunkSize;

    public static void main(String[] args) {
	System.exit(run(args));
    }

    public static int run(String[] args) {

	String videoFilePath;
	String outputFilePathPrefix;
	String frameConverterClassName;

	if (args.length == 3) {
	    videoFilePath = args[0];
	    outputFilePathPrefix = args[1];
	    frameConverterClassName = args[2];
	} else {
	    System.err
		    .println("ERROR: Invalid arguments '"
			    + Arrays.toString(args)
			    + "'.\nUSAGE: AVIConverter videoFilePath (raw AVI) outputFilePathPrefix and frameConverterClassName.");
	    return -1;
	}
	AVIFrameConverter frameConverter;
	try {
	    Class<?> frameConsumerClass = Class
		    .forName(frameConverterClassName);
	    frameConverter = (AVIFrameConverter) frameConsumerClass
		    .newInstance();
	} catch (ClassNotFoundException ex) {
	    System.err
		    .println("ERROR: Cannot instantiate frame converter class '"
			    + frameConverterClassName + "'.");
	    return -2;
	} catch (InstantiationException ex) {
	    System.err
		    .println("ERROR: Cannot instantiate frame converter class '"
			    + frameConverterClassName + "'.");
	    return -2;
	} catch (IllegalAccessException ex) {
	    System.err
		    .println("ERROR: Cannot instantiate frame converter class '"
			    + frameConverterClassName + "'.");
	    return -2;
	}
	AVIConverter aviConverter = new AVIConverter(videoFilePath,
		outputFilePathPrefix, frameConverter);
	aviConverter.run();
	return 0;
    }

    private AVIConverter(String videoFilePath, String outputFilePathPrefix,
	    AVIFrameConverter frameConverter) {

	if (videoFilePath == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'videoFilePath' must not be null.");
	}
	if (outputFilePathPrefix == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'outputFilePathPrefix' must not be null.");
	}
	if (frameConverter == null) {
	    throw new IllegalArgumentException(
		    "Parameter 'frameConverter' must not be null.");
	}

	log = Log.create(System.out, false);

	this.videoFilePath = videoFilePath;
	this.outputFilePathPrefix = outputFilePathPrefix;
	this.frameConverter = frameConverter;

	printFileInfo("Video File", videoFilePath);

    }

    private void run() {
	long startTime = System.currentTimeMillis();
	File videoFile = new File(videoFilePath);
	try {
	    videoInputStream = new FileInputStream(videoFile);
	} catch (IOException ex) {
	    log.log("ERROR: Media file '" + videoFile.getAbsolutePath()
		    + "' cannot be opened. " + ex.getMessage());
	    return;
	}

	try {
	    frameConverter.open(outputFilePathPrefix);
	    parse();
	} catch (IOException ex2) {
	    log.log("ERROR: Output files for path prefix'"
		    + outputFilePathPrefix + "' cannot be opened. "
		    + ex2.getMessage());
	    return;

	} finally {
	    try {
		videoInputStream.close();
	    } catch (IOException ex) {
		throw new RuntimeException("Cannot close file", ex);
	    }

	    frameConverter.close();

	}

	long duration = System.currentTimeMillis() - startTime;
	log.log("Done after " + duration / 1000 + "s.");
	printFileInfo("Output File", outputFilePathPrefix + ".til");

    }

    private void printFileInfo(String title, String filePath) {
	log.printString(title + " Path", filePath);
	log.println();
	long fileSize = new File(filePath).length();
	log.printString(title + " Size", Long.toString(fileSize) + " / 0x"
		+ Long.toHexString(fileSize));
	log.println();
    }

    private void parse() throws IOException {
	parse(videoInputStream);
	frameConverter.logSummary();

    }

    private void parse(InputStream inputStream) throws IOException {
	if (inputStream == null) {
	    throw new IllegalArgumentException(
		    "Parameter inputStream must not be null.");
	}

	AVIReader aviReader = new AVIReader(inputStream);
	LIST aviList = aviReader.readList();
	log.log(aviList.toString());
	parseRIFF(aviReader, true);
	frameConverter.saveHeaders();
    }

    private void parseRIFF(AVIReader aviReader, boolean first)
	    throws IOException {
	if (aviReader == null) {
	    throw new IllegalArgumentException(
		    "Parameter aviReader must not be null.");
	}
	boolean useAudioChunks = false;

	AVIHeader aviHeader = null;
	ATOM atom = null;

	if (first) {
	    LIST hdr1List = aviReader.readList();
	    log.logDetails(hdr1List.toString());
	    aviHeader = aviReader.readHeader();
	    log.log(aviHeader.toString());
	    atom = aviReader.readAtom();
	    while (!atom.fourCC.equals("movi")) {
		aviReader.skipAtomData(atom);
		log.logDetails(atom.toString());
		atom = aviReader.readAtom();
	    }
	} else {
	    LIST list = aviReader.readList();
	    log.logDetails(list.toString());
	    aviReader.readFourCC(); // "movi"
	}

	int imageChunksCount = 0;
	int audioChunksCount = 0;

	List<CHUNK> imageChunks;
	List<CHUNK> audioChunks;
	imageChunks = new ArrayList<CHUNK>();
	audioChunks = new ArrayList<CHUNK>();

	if (first) {
	    mediaAudioChunkSize = frameConverter.init(log, aviHeader);
	}

	atom = aviReader.readAtom();

	// !atom.fourCC.equals("idx1")
	while (atom != null) {
	    log.logDetails("Atom " + atom.toString());

	    if (atom.fourCC.equals("00db") || atom.fourCC.equals("00dc")) {
		CHUNK imageChunk = (CHUNK) atom;
		aviReader.readChunkData(imageChunk);
		if (imageChunk.size > 0) {
		    imageChunks.add(imageChunk);
		    imageChunksCount = imageChunksCount + 1;
		    log.logDetails("Image " + imageChunksCount + ": size="
			    + imageChunk.size);
		}

	    } else if (atom.fourCC.equals("01wb")) {
		CHUNK audioChunk = (CHUNK) atom;
		aviReader.readChunkData(audioChunk);
		audioChunks.add(audioChunk);
		audioChunksCount = audioChunksCount + 1;
		log.logDetails("Audio " + audioChunksCount + ": size="
			+ audioChunk.size);

	    } else if (atom.fourCC.equals("RIFF")) {
		log.log(atom.toString());
		// String avix = aviReader.readFourCC(); // AVIX
		// if ("AVIX".equals(avix)){
		// throw new RuntimeExeption("Ivalid")
		// }
		parseRIFF(aviReader, false);
	    } else {
		aviReader.skipAtomData(atom);
	    }
	    if (!imageChunks.isEmpty()) {

		CHUNK imageChunk = imageChunks.remove(0);
		CHUNK audioChunk = null;
		if (useAudioChunks && !audioChunks.isEmpty()) {

		    audioChunk = audioChunks.remove(0);
		    if (audioChunk.data.length > mediaAudioChunkSize) {
			log.log("Splitting audio chunk");
			byte[] data = audioChunk.data;

			// Shorten first chunk
			audioChunk.data = new byte[mediaAudioChunkSize];
			System.arraycopy(data, 0, audioChunk.data, 0,
				audioChunk.data.length);

			// Put rest into a next chunk
			CHUNK additionalAudioChunk = new CHUNK(
				audioChunk.position);
			additionalAudioChunk.fourCC = audioChunk.fourCC;
			additionalAudioChunk.size = audioChunk.size
				- mediaAudioChunkSize;
			additionalAudioChunk.data = new byte[data.length
				- mediaAudioChunkSize];
			System.arraycopy(data, 0, additionalAudioChunk.data, 0,
				additionalAudioChunk.data.length);
			audioChunks.add(0, additionalAudioChunk);

			audioChunksCount = audioChunksCount + 1;
			log.log("Audio " + audioChunksCount + ": size="
				+ audioChunk.size);
		    }
		}
		frameConverter.consume(imageChunksCount, imageChunk,
			audioChunksCount, audioChunk);
	    }

	    atom = aviReader.readAtom();
	}

    }
}